// popup.js - V15.0 Final
let pageData = null;
let solutionsLog = [];

document.addEventListener("DOMContentLoaded", () => {
  initTabs();
  loadHistory();
  
  const startBtn = document.getElementById("startScanBtn");
  if (startBtn) {
    startBtn.addEventListener("click", () => {
      document.getElementById("start-screen").classList.add("hidden");
      document.getElementById("loading-screen").classList.remove("hidden");
      startAnalysis();
    });
  }

  document.addEventListener("click", (e) => {
    if (e.target.dataset.highlight) sendHighlightCommand(e.target.dataset.highlight);
  });

  document.getElementById("exportBtn").addEventListener("click", exportReport);
  document.getElementById("clearHistoryBtn")?.addEventListener("click", clearHistory);
});

function initTabs() {
  document.querySelectorAll(".tab-link").forEach(tab => {
    tab.addEventListener("click", () => {
      document.querySelectorAll(".tab-link").forEach(t => t.classList.remove("active"));
      document.querySelectorAll(".tab-content").forEach(c => c.classList.add("hidden"));
      tab.classList.add("active");
      const target = document.getElementById(tab.dataset.tab);
      if(target) target.classList.remove("hidden");
    });
  });
}

async function startAnalysis() {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  if (!tab || !tab.url.startsWith("http")) {
    document.getElementById("loading-screen").innerHTML = "<p style='color:red; text-align:center; padding:20px;'>Hata: Sadece web sayfaları.</p>";
    return;
  }

  try {
    await chrome.scripting.executeScript({ target: { tabId: tab.id }, files: ["ai.js", "analyzer.js"] });

    chrome.tabs.sendMessage(tab.id, { action: "GET_PAGE_DATA" }, (response) => {
      if (chrome.runtime.lastError || !response || response.error) {
        document.getElementById("loading-screen").innerHTML = "<p style='text-align:center;'>Veri alınamadı.<br>Sayfayı yenileyip tekrar deneyin.</p>";
        return;
      }
      
      pageData = response;
      
      document.getElementById("loading-screen").classList.add("hidden");
      document.getElementById("app-header").classList.remove("hidden");
      document.getElementById("tabs-nav").classList.remove("hidden");
      document.getElementById("content-area").classList.remove("hidden");

      const scores = calculateDualScore(response);
      pageData.scores = scores;

      renderData(response, scores);
      generateSolutions(response);
      saveToHistory(response, scores.final);

      const query = (response.topKeywords?.[0]?.word) || (response.title ? response.title.split(" ")[0] : "");
      if(query) fetchGoogleSuggestions(query);
    });
  } catch (e) {
    console.error(e);
    document.getElementById("loading-screen").innerHTML = "<p>Beklenmedik bir hata.</p>";
  }
}

function calculateDualScore(data) {
  let techScore = 100; 
  let aiPenalty = 0;   

  if ((data.titleLength || 0) < 30) techScore -= 5;
  if ((data.titleLength || 0) > 65) techScore -= 5;
  if ((data.descLength || 0) === 0) techScore -= 10;
  else if ((data.descLength || 0) < 100) techScore -= 5;
  
  if (data.headings?.h1?.count !== 1) techScore -= 15;
  if ((data.images?.missingAlt || 0) > 0) techScore -= 5;
  if ((data.links?.unsafe || 0) > 0) techScore -= 10;
  if ((data.wordCount || 0) < 300) techScore -= 10;
  if ((data.cwv?.lcp || 0) > 2500) techScore -= 10;
  
  const aiScore = data.aiAnalysis?.score || 0;
  if (aiScore > 70) aiPenalty = 10;      
  else if (aiScore > 50) aiPenalty = 5; 

  techScore = Math.max(0, techScore);
  const finalScore = Math.max(0, techScore - aiPenalty);

  return { technical: techScore, final: finalScore, penalty: aiPenalty };
}

function saveToHistory(data, score) {
  const item = {
    url: data.domain,
    date: new Date().toLocaleTimeString("tr-TR", {hour: '2-digit', minute:'2-digit'}),
    score: score
  };
  chrome.storage.local.get("seoHistory", (result) => {
    let history = result.seoHistory || [];
    history.unshift(item);
    if (history.length > 10) history.pop();
    chrome.storage.local.set({ seoHistory: history }, loadHistory);
  });
}

function loadHistory() {
  chrome.storage.local.get("seoHistory", (result) => {
    const list = document.getElementById("history-list");
    if (!list) return;
    const history = result.seoHistory || [];
    if (history.length === 0) {
      list.innerHTML = "<li style='padding:10px; color:#999;'>Henüz kayıt yok.</li>";
      return;
    }
    list.innerHTML = history.map(h => {
      const color = h.score > 80 ? "#22c55e" : h.score > 50 ? "#f59e0b" : "#ef4444";
      return `<li style="display:flex; justify-content:space-between; align-items:center; padding:8px; border-bottom:1px solid #eee;">
          <div><div style="font-weight:bold; font-size:12px; color:#334155;">${h.url}</div><div style="font-size:10px; color:#94a3b8;">${h.date}</div></div>
          <div style="background:${color}; color:white; padding:2px 8px; border-radius:10px; font-weight:bold; font-size:11px;">${h.score}</div>
        </li>`;
    }).join("");
  });
}

function clearHistory() { chrome.storage.local.remove("seoHistory", loadHistory); }

function renderData(data, scores) {
  const sBadge = document.getElementById("overallScore");
  if(sBadge) {
    sBadge.innerText = scores.final;
    sBadge.style.background = scores.final > 80 ? "#22c55e" : scores.final > 50 ? "#f59e0b" : "#ef4444";
    sBadge.title = `Teknik SEO: ${scores.technical} | AI Cezası: -${scores.penalty}`;
  }

  setText("ov-url", data.domain);
  setText("ov-robots", data.robots);
  setText("ov-canonical", data.isCanonicalSelf ? "Self" : (data.canonical ? "Farklı" : "Yok"));
  
  setText("serp-title", data.title || "Başlık Yok");
  setText("serp-desc", data.description || "Açıklama Yok");
  setText("serp-url", data.domain);

  const aiData = data.aiAnalysis || {};
  setTimeout(() => {
    const bar = document.getElementById("ai-bar");
    if(bar) { bar.style.width = (aiData.score||0) + "%"; bar.style.background = aiData.color; }
  }, 100);
  setText("ai-label", `%${aiData.score||0} - ${aiData.label||"-"}`);
  setText("ai-std", aiData.details);
  
  setText("word-count", data.wordCount);
  setText("text-ratio", "%" + (data.textRatio||0));
  setText("readability-score", data.readability?.level);

  // N-GRAMS RENDER (GÜNCELLENDİ)
  const renderKw = (id, list) => {
    const el = document.getElementById(id);
    if(el) el.innerHTML = (list && list.length) ? list.map(k=>`<span class="keyword-tag">${k.word} (${k.count})</span>`).join("") : "-";
  };
  renderKw("top-keywords", data.topKeywords); // Tekil
  renderKw("kw-double", data.keywords?.double); // İkili
  renderKw("kw-triple", data.keywords?.triple); // Üçlü

  const hList = document.getElementById("headings-list");
  if(hList) hList.innerHTML = `<li>H1 (${data.headings?.h1?.count||0}): ${(data.headings?.h1?.texts||[]).slice(0,2).join(" | ")}</li><li>H2 (${data.headings?.h2?.count||0})</li>`;

  const tList = document.getElementById("tech-stack-list");
  if(tList) tList.innerHTML = (data.techStack && data.techStack.length) ? data.techStack.map(t=>`<span class="badge ok">${t}</span>`).join(" ") : "-";
  
  const speedVal = data.cwv?.lcp || 0;
  const speedLabel = data.cwv?.label || "LCP";
  setText("cwv-lcp", speedVal > 0 ? `${speedVal} ms (${speedLabel})` : "Ölçülemedi");
  setText("cwv-cls", data.cwv?.cls);
  
  const sc = document.getElementById("schema-status");
  if(sc) {
    if(data.schemaTypes && data.schemaTypes.length > 0) {
        sc.innerText = `✅ ${data.schemaTypes.length} Adet (${data.schemaTypes[0]})`;
        sc.style.color = "green";
    } else { sc.innerText = "❌ Yok"; sc.style.color = "red"; }
  }

  setText("img-missing", data.images?.missingAlt);
  setText("link-unsafe", data.links?.unsafe);
  setText("img-format", data.imgAnalysis?.notModern);
  setText("img-size", data.imgAnalysis?.tooLarge);
  setText("a11y-font", data.a11y?.smallFont);
  setText("a11y-target", data.a11y?.smallTargets);
}

function generateSolutions(data) {
  const list = document.getElementById("suggestion-list");
  if(!list) return;
  list.innerHTML = "";
  solutionsLog = [];

  function add(type, title, fix) {
    const li = document.createElement("li");
    li.className = `suggestion-item ${type}`;
    li.innerHTML = `<div class="s-header"><strong>${title}</strong></div><div class="s-fix">💡 ${fix}</div>`;
    list.appendChild(li);
    solutionsLog.push({ type, title, fix });
  }

  // Canonical Uyarısı (YENİ)
  if (!data.isCanonicalSelf && data.canonical !== "") {
    add("warn", "Canonical Uyuşmazlığı", `Sayfa başka bir URL'i işaret ediyor: ${data.canonical}. Bu sayfa indexlenmeyebilir.`);
  }

  if((data.titleLength||0)<30) add("warn","Başlık Kısa", "30-60 karakter yapın.");
  if((data.descLength||0)===0) add("error","Meta Açıklama Yok", "Description ekleyin.");
  if((data.images?.missingAlt||0)>0) add("warn","Eksik Alt", "Görsellere açıklama girin.");
  if((data.aiAnalysis?.score||0)>70) add("warn","Yüksek AI Oranı", "İçerik monoton. Doğallaştırın.");
  if((data.cwv?.lcp||0)>2500) add("warn","Yavaş Site", "LCP 2.5sn üzerinde. Hızlandırın.");
  
  if(solutionsLog.length === 0) list.innerHTML = "<li class='suggestion-item good'>🎉 Sorun Yok!</li>";
}

function fetchGoogleSuggestions(query) {
  const list = document.getElementById("google-suggestions");
  if(!list) return;
  list.innerHTML = "<li><i>Yükleniyor...</i></li>";
  chrome.runtime.sendMessage({ action: "FETCH_AUTOCOMPLETE", query }, (res) => {
    if (res && res.success && res.suggestions.length) {
      list.innerHTML = res.suggestions.slice(0, 5).map(s => `<li>🔎 ${s}</li>`).join("");
    } else list.innerHTML = "<li>Öneri bulunamadı.</li>";
  });
}

function setText(id, t) { const e=document.getElementById(id); if(e) e.innerText=(t!==undefined&&t!==null)?t:"-"; }
function sendHighlightCommand(s) { chrome.tabs.query({active:true,currentWindow:true}, t=>chrome.tabs.sendMessage(t[0].id,{action:"HIGHLIGHT",selector:s})); }
function exportReport() { if(!pageData)return; pageData.solutionsLog=solutionsLog; chrome.storage.local.set({'seoReportData':pageData}).then(()=>chrome.tabs.create({url:'report.html'})); }