// analyzer.js - V15.0 (N-Gram Added, Social Removed)
(function () {
  if (window.hasRunSeoAnalyzer) return;
  window.hasRunSeoAnalyzer = true;

  chrome.runtime.onMessage.addListener((req, sender, sendResponse) => {
    if (req.action === "HIGHLIGHT") {
      document.querySelectorAll(".seo-highlight").forEach(el => {
        el.style.outline = ""; el.style.boxShadow = ""; el.classList.remove("seo-highlight");
      });
      if (req.selector) {
        const elements = document.querySelectorAll(req.selector);
        elements.forEach(el => {
          el.style.outline = "4px solid #ef4444";
          el.style.boxShadow = "0 0 15px rgba(239, 68, 68, 0.6)";
          el.scrollIntoView({ behavior: "smooth", block: "center" });
          el.classList.add("seo-highlight");
        });
      }
    }
    if (req.action === "GET_PAGE_DATA") {
      try {
        const data = collectData();
        sendResponse(data);
      } catch (e) {
        console.error("Analiz Hatası:", e);
        // Hata olsa bile popup'ın çökmemesi için boş şablon dön
        sendResponse({ error: true, msg: e.message, url: window.location.href });
      }
    }
  });

  // --- N-GRAM ANALİZİ (2'li ve 3'lü Kelime Öbekleri) ---
  function getNGrams(text, n) {
    const stopwords = ["ve","veya","bir","bu","ile","için","çok","da","de","o","şu","ama","fakat","gibi","daha","en","mi","mu","ben","sen","biz","var","yok","olan","olarak","nedir","ne","nasıl"];
    const words = text.toLowerCase().replace(/[^\p{L}\s]/gu, "").split(/\s+/).filter(w => w.length > 2);
    
    const ngrams = {};
    for (let i = 0; i < words.length - n + 1; i++) {
      const phrase = words.slice(i, i + n).join(" ");
      // Stopword kontrolü eklenebilir ama basit tutuyoruz
      ngrams[phrase] = (ngrams[phrase] || 0) + 1;
    }
    
    return Object.entries(ngrams)
      .sort((a,b) => b[1] - a[1]) // Çoktan aza sırala
      .slice(0, 5) // İlk 5 tanesini al
      .map(i => ({ word: i[0], count: i[1] }));
  }

  // --- YARDIMCI FONKSİYONLAR ---

  function getCWV() {
    let lcp = 0;
    let cls = 0;
    let label = "LCP";

    try {
      const clsEntries = performance.getEntriesByType("layout-shift");
      if (clsEntries) clsEntries.forEach(e => { if (!e.hadRecentInput) cls += e.value; });

      const lcpEntries = performance.getEntriesByType("largest-contentful-paint");
      if (lcpEntries && lcpEntries.length > 0) {
        const lastEntry = lcpEntries[lcpEntries.length - 1];
        lcp = Math.round(lastEntry.renderTime || lastEntry.loadTime);
      } 
      
      if (lcp === 0 && window.performance.timing) {
        const timing = window.performance.timing;
        const loadTime = timing.loadEventEnd - timing.navigationStart;
        if (loadTime > 0) {
          lcp = loadTime;
          label = "Load";
        }
      }
    } catch (e) { console.log("Perf hatası", e); }

    return { lcp, cls: cls.toFixed(3), label };
  }

  function detectTechStack() {
    const stack = [];
    try {
      const html = document.documentElement.outerHTML.toLowerCase();
      const scripts = Array.from(document.scripts).map(s => (s.src || "").toLowerCase());
      const metas = Array.from(document.querySelectorAll("meta")).map(m => (m.content || "").toLowerCase());

      if (html.includes("wp-content") || metas.some(m => m.includes("wordpress"))) stack.push("WordPress");
      if (html.includes("shopify")) stack.push("Shopify");
      if (html.includes("wix.com")) stack.push("Wix");
      if (scripts.some(s => s.includes("gtm.js"))) stack.push("GTM");
      if (scripts.some(s => s.includes("analytics.js") || s.includes("gtag"))) stack.push("Analytics");
      if (html.includes("react") || scripts.some(s=>s.includes("react"))) stack.push("React");
      if (scripts.some(s => s.includes("jquery"))) stack.push("jQuery");
      if (scripts.some(s => s.includes("bootstrap"))) stack.push("Bootstrap");
    } catch(e) {}
    return [...new Set(stack)];
  }

  function detectSchema() {
    const schemas = [];
    document.querySelectorAll('script[type="application/ld+json"]').forEach(s => {
      try {
        const j = JSON.parse(s.innerText);
        if(j["@type"]) {
            const types = Array.isArray(j["@type"]) ? j["@type"] : [j["@type"]];
            schemas.push(...types);
        } else if (j["@graph"]) {
            j["@graph"].forEach(g => { if(g["@type"]) schemas.push(g["@type"]); });
        }
      } catch(e){}
    });
    document.querySelectorAll('[itemtype]').forEach(el => {
        const type = el.getAttribute('itemtype');
        if(type) schemas.push(type.split('/').pop());
    });
    return [...new Set(schemas)];
  }

  function calculateReadability(text) {
    if (!text) return { score: 0, level: "-" };
    const sentences = text.split(/[.!?]+/).filter(Boolean);
    const words = text.split(/\s+/).filter(Boolean);
    const avg = words.length / (sentences.length || 1);
    let score = 100 - (avg * 1.5);
    score = Math.max(0, Math.min(100, Math.round(score)));
    const level = score > 80 ? "Çok Kolay" : score > 60 ? "Kolay" : score > 40 ? "Orta" : "Zor";
    return { score, level };
  }

  function getTopKeywords(text) {
    const stopwords = ["ve","veya","bir","bu","ile","için","çok","da","de","o","şu","ama","fakat","gibi","daha","en","mi","mu","ben","sen","biz","var","yok","olan","olarak","nedir"];
    const words = text.toLowerCase().replace(/[^\p{L}\s]/gu, "").split(/\s+/).filter(w => w.length > 2 && !stopwords.includes(w));
    const freq = {};
    words.forEach(w => freq[w] = (freq[w] || 0) + 1);
    return Object.entries(freq).sort((a,b) => b[1] - a[1]).slice(0, 5).map(i => ({ word: i[0], count: i[1] }));
  }

  // --- ANA COLLECT DATA ---
  function collectData() {
    const data = {};
    
    data.url = window.location.href;
    data.domain = window.location.hostname;
    data.title = document.title || "";
    data.titleLength = data.title.length;
    data.description = document.querySelector('meta[name="description"]')?.content || "";
    data.descLength = data.description.length;
    data.robots = document.querySelector('meta[name="robots"]')?.content || "index, follow";
    data.canonical = document.querySelector('link[rel="canonical"]')?.href || "";
    data.isCanonicalSelf = data.canonical === data.url;

    const bodyText = document.body.innerText || "";
    const cleanText = bodyText.replace(/\s+/g, " ").trim();
    data.wordCount = cleanText.split(" ").length;
    
    const htmlLen = document.documentElement.outerHTML.length;
    data.textRatio = htmlLen > 0 ? Math.round((bodyText.length / htmlLen) * 100) : 0;

    if (window.AI_Analyzer) {
      data.aiAnalysis = window.AI_Analyzer.detect(cleanText);
    } else {
      data.aiAnalysis = { score: 0, label: "AI Modülü Yok", color: "#ccc", details: "-" };
    }

    data.readability = calculateReadability(cleanText);
    
    // N-GRAMS (YENİ) - Tekil zaten vardı, 2'li ve 3'lü eklendi
    data.topKeywords = getTopKeywords(cleanText); // Mevcut yapı korundu
    data.keywords = {
      double: getNGrams(cleanText, 2), // İkili öbekler
      triple: getNGrams(cleanText, 3)  // Üçlü öbekler
    };

    data.techStack = detectTechStack();
    data.cwv = getCWV();
    data.schemaTypes = detectSchema();

    data.images = { total: 0, missingAlt: 0 };
    data.imgAnalysis = { notModern: 0, tooLarge: 0 };
    
    const resources = performance.getEntriesByType("resource");

    document.querySelectorAll("img").forEach(img => {
      if(img.width > 20 && img.height > 20) {
        data.images.total++;
        if(!img.alt || img.alt.trim() === "") data.images.missingAlt++;
        
        const src = img.currentSrc || img.src;
        if (!src.includes(".webp") && !src.includes(".avif") && !src.includes(".svg")) data.imgAnalysis.notModern++;
        
        const res = resources.find(r => r.name === src);
        if (res && res.transferSize > 150000) data.imgAnalysis.tooLarge++; 
      }
    });

    data.links = { total: 0, internal: 0, external: 0, unsafe: 0 };
    document.querySelectorAll("a[href]").forEach(a => {
      data.links.total++;
      try {
        const u = new URL(a.href, document.baseURI);
        if(u.hostname === window.location.hostname) data.links.internal++;
        else {
          data.links.external++;
          if(a.target === "_blank" && !a.rel.includes("noopener") && !a.rel.includes("noreferrer")) data.links.unsafe++;
        }
      } catch(e){}
    });

    data.headings = { h1: {count:0, texts:[]}, h2: {count:0}, h3: {count:0} };
    ["h1", "h2", "h3"].forEach(tag => {
      const els = Array.from(document.querySelectorAll(tag));
      data.headings[tag] = { count: els.length, texts: els.map(e => e.innerText.trim()).filter(Boolean) };
    });

    data.a11y = { smallFont: 0 };
    document.querySelectorAll("p, span, li").forEach(el => {
       if(el.style.fontSize && parseFloat(el.style.fontSize) < 12) data.a11y.smallFont++;
    });

    return data;
  }
})();