// ai.js – Türkçe AI Tespit Aracı (v6 – strict, kurumsal metin odaklı)
// Heuristik / istatistik tabanlı, tarayıcıda çalışan agresif dedektör.

window.AI_Analyzer = (function () {

  const AI_MARKERS = [
    "sonuç olarak",
    "genel olarak",
    "özetle",
    "kısacası",
    "nihayetinde",
    "bu bağlamda",
    "bu doğrultuda",
    "bu kapsamda",
    "buna ek olarak",
    "bununla birlikte",
    "diğer yandan",
    "öte yandan",
    "ayrıca",
    "dahası",
    "ilk olarak",
    "ikinci olarak",
    "son olarak",
    "özellikle",
    "günümüzde",
    "modern dünyada",
    "önemli bir rol oynamaktadır",
    "önemli bir rol oynar",
    "büyük ölçüde",
    "kayda değer ölçüde",
    "dikkate alındığında",
    "göz önüne alındığında",
    "geniş bir perspektiften bakıldığında",
    "bir diğer önemli nokta",
    "bunun sonucunda",
    "bu nedenle",
    "bu sebeple",
    "dolayısıyla",
    "bir başka deyişle",
    "diğer bir deyişle",
    "temel olarak"
  ];

  const FORMAL_OPENINGS = [
    "bu çalışma",
    "bu makale",
    "bu yazıda",
    "bu incelemede",
    "bu raporda",
    "bu araştırma",
    "bu bağlamda"
  ];

  const FUNCTION_WORDS = [
    "ve","de","da","ki","bu","şu","o","bir","çok","az","gibi",
    "ile","için","ama","fakat","ancak","çünkü","ya","veya","ya da",
    "hem","ne","mi","mu","mü","ise","şayet","dahi","bile","yani",
    "mı","mu","mü","üzere","göre","daha","en"
  ];

  function clamp(v, min, max) {
    return Math.max(min, Math.min(max, v));
  }

  function charEntropy(str) {
    const cleaned = str.toLowerCase().replace(/[^a-zçğıöşüı0-9\s.,;:!?]/g, "");
    if (!cleaned) return 0;
    const counts = {};
    let total = 0;
    for (let ch of cleaned) {
      if (ch === " ") continue;
      counts[ch] = (counts[ch] || 0) + 1;
      total++;
    }
    let H = 0;
    Object.values(counts).forEach(c => {
      const p = c / total;
      H -= p * Math.log2(p);
    });
    return H;
  }

  // Başlık gibi görünen satırları say (kurumsal/SEO metin tespiti için)
  function detectHeadings(text) {
    const lines = text.split(/\r?\n/).map(l => l.trim()).filter(l => l.length > 0);
    let headingCount = 0;
    lines.forEach(line => {
      // Çok kısa veya çok uzun satırları ele
      const wordCount = line.split(/\s+/).length;
      if (wordCount < 2 || wordCount > 12) return;
      // Satırın sonunda nokta, soru işareti vs. yoksa ve baş harfler büyükse başlık adayı say
      if (!/[.!?]$/.test(line) && /^[A-ZÇĞİÖŞÜ]/.test(line)) {
        headingCount++;
      }
    });
    return {
      headingCount,
      totalLines: lines.length,
      headingDensity: lines.length > 0 ? headingCount / lines.length : 0
    };
  }

  function calculateStats(text) {
    const normalized = text.replace(/\s+/g, " ").trim();
    if (!normalized) return null;

    const sentences = normalized
      .split(/[.!?…]+/u)
      .map(s => s.trim())
      .filter(s => s.length > 0);

    const rawWords = normalized
      .toLowerCase()
      .replace(/[^\p{L}\s]/gu, " ")
      .split(/\s+/)
      .filter(w => w.length > 0);

    const totalWords = rawWords.length;
    if (totalWords < 60) return null;

    const sentenceLengths = sentences.map(s =>
      s
        .toLowerCase()
        .replace(/[^\p{L}\s]/gu, " ")
        .split(/\s+/)
        .filter(w => w.length > 0).length
    );
    const sentenceCount = sentenceLengths.length;
    const avgSentenceLength =
      sentenceLengths.reduce((a, b) => a + b, 0) / sentenceLengths.length;

    const variance =
      sentenceLengths.reduce(
        (sum, n) => sum + Math.pow(n - avgSentenceLength, 2),
        0
      ) / sentenceLengths.length;
    const stdDev = Math.sqrt(variance);

    const lower = normalized.toLowerCase();
    let markerCount = 0;
    AI_MARKERS.forEach(m => {
      if (lower.includes(m)) markerCount++;
    });
    const markerDensity = (markerCount / totalWords) * 100;

    const uniqueWordCount = new Set(rawWords).size;
    const ttr = uniqueWordCount / totalWords;

    const totalChars = rawWords.reduce((sum, w) => sum + w.length, 0);
    const avgWordLength = totalChars / totalWords;

    const punctuationMatches = text.match(/[.,;:!?]/g) || [];
    const punctuationCount = punctuationMatches.length;
    const commaCount = (text.match(/,/g) || []).length;
    const exqCount = (text.match(/[!?]/g) || []).length;
    const punctuationDensity = punctuationCount / totalWords;
    const commaDensity = commaCount / totalWords;
    const exqRatio = punctuationCount > 0 ? (exqCount / punctuationCount) : 0;

    let formalOpeningHits = 0;
    FORMAL_OPENINGS.forEach(o => {
      if (lower.startsWith(o)) formalOpeningHits++;
    });

    const bigramCounts = {};
    for (let i = 0; i < rawWords.length - 1; i++) {
      const bg = rawWords[i] + " " + rawWords[i + 1];
      bigramCounts[bg] = (bigramCounts[bg] || 0) + 1;
    }
    const repeatedBigrams = Object.values(bigramCounts).filter(c => c > 2).length;
    const bigramRepetitionRate =
      repeatedBigrams / Math.max(1, Object.keys(bigramCounts).length);

    let functionCount = 0;
    rawWords.forEach(w => {
      if (FUNCTION_WORDS.includes(w)) functionCount++;
    });
    const functionWordRate = functionCount / totalWords;

    const paragraphs = text
      .split(/\n\s*\n+/)
      .map(p => p.trim())
      .filter(p => p.length > 0);

    let paragraphStdDev = 0;
    if (paragraphs.length > 1) {
      const pLens = paragraphs.map(p =>
        p
          .toLowerCase()
          .replace(/[^\p{L}\s]/gu, " ")
          .split(/\s+/)
          .filter(w => w.length > 0).length
      );
      const pAvg = pLens.reduce((a, b) => a + b, 0) / pLens.length;
      const pVar =
        pLens.reduce((s, n) => s + Math.pow(n - pAvg, 2), 0) / pLens.length;
      paragraphStdDev = Math.sqrt(pVar);
    }

    const H = charEntropy(text);

    // Yeni: başlık ve soru oranları
    const headingInfo = detectHeadings(text);
    const headingDensity = headingInfo.headingDensity;

    const questions = (text.match(/\?/g) || []).length;
    const questionDensity = questions / sentenceCount;

    return {
      totalWords,
      sentenceCount,
      stdDev,
      markerDensity,
      ttr,
      avgSentenceLength,
      avgWordLength,
      punctuationDensity,
      commaDensity,
      exqRatio,
      formalOpeningHits,
      bigramRepetitionRate,
      functionWordRate,
      paragraphStdDev,
      charEntropy: H,
      headingCount: headingInfo.headingCount,
      headingDensity,
      questionDensity
    };
  }

  function featureScores(stats) {
    const scores = {};
    const comments = [];

    // 1) Cümle varyasyonu
    let s_std;
    if (stats.stdDev <= 3)      { s_std = 1;   comments.push("Cümle uzunlukları aşırı düzenli (çok güçlü AI sinyali)."); }
    else if (stats.stdDev <= 5) { s_std = 0.7; comments.push("Cümle uzunlukları oldukça düzenli (AI lehine)."); }
    else if (stats.stdDev <= 7) { s_std = 0.2; comments.push("Cümle uzunlukları orta düzeyde düzenli (hafif AI)."); }
    else if (stats.stdDev <= 11){ s_std = -0.2;comments.push("Cümle uzunlukları dağınık (insan lehine)."); }
    else                        { s_std = -0.6;comments.push("Cümle uzunlukları çok dağınık (güçlü insan sinyali)."); }
    scores.stdDev = s_std;

    // 2) AI kalıp yoğunluğu
    let s_marker;
    if      (stats.markerDensity >= 2.5) { s_marker = 1;   comments.push("AI kalıpları çok yüksek yoğunlukta."); }
    else if (stats.markerDensity >= 1.2) { s_marker = 0.8; comments.push("AI kalıpları bariz seviyede kullanılmış."); }
    else if (stats.markerDensity >= 0.6) { s_marker = 0.4; comments.push("AI kalıpları az da olsa mevcut."); }
    else if (stats.markerDensity >= 0.2) { s_marker = 0;   comments.push("AI kalıpları düşük seviyede; belirleyici değil."); }
    else                                 { s_marker = -0.3;comments.push("AI kalıbı neredeyse yok (insan lehine)."); }
    scores.markerDensity = s_marker;

    // 3) Kelime çeşitliliği (TTR)
    let s_ttr;
    if      (stats.ttr >= 0.70) { s_ttr = 0.4; comments.push("Kelime çeşitliliği çok yüksek (AI lehine)."); }
    else if (stats.ttr >= 0.60) { s_ttr = 0.25;comments.push("Kelime çeşitliliği yüksek (hafif AI)."); }
    else if (stats.ttr >= 0.50) { s_ttr = 0.1; comments.push("Kelime çeşitliliği orta (nötr)."); }
    else if (stats.ttr >= 0.42) { s_ttr = -0.3;comments.push("Kelime çeşitliliği düşük (insan lehine)."); }
    else                        { s_ttr = -0.6;comments.push("Kelime çeşitliliği çok düşük (güçlü insan sinyali)."); }
    scores.ttr = s_ttr;

    // 4) Ortalama cümle uzunluğu
    let s_avgSent;
    if      (stats.avgSentenceLength >= 18 && stats.avgSentenceLength <= 24) {
      s_avgSent = 0.9; comments.push("Cümleler 18–24 kelime bandında, tipik AI yapısı.");
    } else if (stats.avgSentenceLength >= 14 && stats.avgSentenceLength < 18) {
      s_avgSent = 0.4; comments.push("Cümle uzunlukları AI tarzına yakın.");
    } else if (stats.avgSentenceLength > 24 && stats.avgSentenceLength <= 30) {
      s_avgSent = 0.4; comments.push("Uzun ve yapılandırılmış cümleler (AI olabilir).");
    } else if (stats.avgSentenceLength >= 9 && stats.avgSentenceLength <= 14) {
      s_avgSent = -0.2;comments.push("Cümleler kısa/konuşma dili gibi (insan lehine).");
    } else {
      s_avgSent = -0.5;comments.push("Cümle uzunlukları çok uç; serbest yazım (insan lehine).");
    }
    scores.avgSentenceLength = s_avgSent;

    // 5) Ortalama kelime uzunluğu
    let s_avgWord;
    if      (stats.avgWordLength >= 5.4) { s_avgWord = 0.85;comments.push("Kelime uzunlukları yüksek; teknik/soyut yoğun (AI lehine)."); }
    else if (stats.avgWordLength >= 5.0) { s_avgWord = 0.45;comments.push("Kelime uzunlukları ortalamanın üzerinde (AI lehine hafif)."); }
    else if (stats.avgWordLength >= 4.4) { s_avgWord = 0.1; comments.push("Kelime uzunlukları normal aralıkta (nötr)."); }
    else if (stats.avgWordLength >= 3.8) { s_avgWord = -0.3;comments.push("Kelime uzunlukları kısa; sade dil (insan lehine)."); }
    else                                 { s_avgWord = -0.6;comments.push("Kelime uzunlukları çok kısa; gündelik dil (insan lehine güçlü)."); }
    scores.avgWordLength = s_avgWord;

    // 6) Virgül yoğunluğu
    let s_comma;
    if      (stats.commaDensity >= 0.08) { s_comma = 0.75;comments.push("Virgül yoğun; kontrollü alt cümleler (AI lehine)."); }
    else if (stats.commaDensity >= 0.05) { s_comma = 0.4; comments.push("Virgül kullanımı belirgin (hafif AI)."); }
    else if (stats.commaDensity >= 0.02) { s_comma = 0.1; comments.push("Virgül kullanımı normal (nötr)."); }
    else if (stats.commaDensity >= 0.01) { s_comma = -0.2;comments.push("Virgül az; basit cümleler (insan lehine)."); }
    else                                 { s_comma = -0.4;comments.push("Virgül neredeyse yok; çok basit yapı (insan lehine)."); }
    scores.commaDensity = s_comma;

    // 7) Formal açılış
    let s_formal = 0;
    if (stats.formalOpeningHits > 0) {
      s_formal = 0.8;
      comments.push("Metin resmi/makale tarzı girişle başlıyor (AI lehine).");
    }
    scores.formalOpening = s_formal;

    // 8) Bigram tekrar oranı
    let s_bigram;
    if      (stats.bigramRepetitionRate >= 0.15) { s_bigram = 0.75;comments.push("İkili kelime kalıpları sık tekrarlanmış (şablon yapı, AI lehine)."); }
    else if (stats.bigramRepetitionRate >= 0.08) { s_bigram = 0.35;comments.push("Bazı ikili kelime kalıpları tekrar ediyor (AI lehine hafif)."); }
    else if (stats.bigramRepetitionRate >= 0.03) { s_bigram = 0;   comments.push("İkili kelime tekrar oranı doğal (nötr)."); }
    else                                         { s_bigram = -0.2;comments.push("İkili kelime tekrarları çok az (insan lehine)."); }
    scores.bigramRepetitionRate = s_bigram;

    // 9) İşlevsel kelime oranı
    let s_func;
    if      (stats.functionWordRate <= 0.32) { s_func = 0.9; comments.push("İşlevsel kelimeler düşük; içerik kelimesi baskın (güçlü AI sinyali)."); }
    else if (stats.functionWordRate <= 0.38) { s_func = 0.5; comments.push("İşlevsel kelime oranı düşük (AI lehine)."); }
    else if (stats.functionWordRate <= 0.46) { s_func = 0.1; comments.push("İşlevsel kelime oranı normal (nötr)."); }
    else if (stats.functionWordRate <= 0.55) { s_func = -0.3;comments.push("İşlevsel kelime oranı yüksek (insan lehine)."); }
    else                                     { s_func = -0.6;comments.push("İşlevsel kelime oranı çok yüksek; konuşma dili (insan lehine güçlü)."); }
    scores.functionWordRate = s_func;

    // 10) Karakter entropisi
    let s_ent = 0;
    if (stats.charEntropy >= 3.4 && stats.charEntropy <= 4.0) {
      s_ent = 0.35;
      comments.push("Karakter entropisi orta bandta; model çıktısına yakın.");
    } else if (stats.charEntropy > 4.3) {
      s_ent = -0.3;
      comments.push("Karakter entropisi yüksek; dağınık insan yazımı olabilir.");
    }
    scores.charEntropy = s_ent;

    // 11) Noktalama profili
    let s_punc = 0;
    if (stats.exqRatio < 0.05 && stats.punctuationDensity > 0.03) {
      s_punc = 0.35;
      comments.push("Noktalama ciddi ama ! ve ? çok az; resmi/duygusuz (AI lehine).");
    } else if (stats.exqRatio > 0.25) {
      s_punc = -0.3;
      comments.push("! ve ? bariz kullanılmış; duygusal/konuşma dili (insan lehine).");
    }
    scores.punctuationProfile = s_punc;

    // 12) Paragraf simetrisi
    let s_para = 0;
    if (stats.paragraphStdDev > 0 && stats.paragraphStdDev < 10) {
      s_para = 0.35;
      comments.push("Paragraf uzunlukları birbirine çok yakın; düzenli yapı (AI lehine).");
    } else if (stats.paragraphStdDev > 25) {
      s_para = -0.3;
      comments.push("Paragraf uzunlukları çok değişken; insan lehine.");
    }
    scores.paragraphSymmetry = s_para;

    // 13) Başlık yoğunluğu
    let s_head = 0;
    if (stats.headingDensity >= 0.20) {
      s_head = 0.7;
      comments.push("Metinde çok sayıda başlık / alt başlık var (kurumsal/AI yapı).");
    } else if (stats.headingDensity >= 0.10) {
      s_head = 0.35;
      comments.push("Metinde belirgin düzeyde başlık kullanımı var.");
    }
    scores.headingDensity = s_head;

    // 14) Soru yoğunluğu (FAQ / Q&A pattern)
    let s_q = 0;
    if (stats.questionDensity >= 0.10) {
      s_q = 0.5;
      comments.push("Metinde sık soru–cevap kalıbı kullanılmış (kurumsal/AI içerik stili).");
    }
    scores.questionDensity = s_q;

    return { scores, comments };
  }

  function detect(text, options = { mode: "strict" }) {
    const stats = calculateStats(text);

    if (!stats) {
      return {
        score: 0,
        rawScore: 0,
        label: "Yetersiz Veri",
        color: "#94a3b8",
        mode: options.mode || "strict",
        details: "Analiz için en az 60 kelimelik bir Türkçe metin girin.",
        stats: null,
        raw: null
      };
    }

    const { scores, comments } = featureScores(stats);

    const weights = {
      stdDev: 0.16,
      markerDensity: 0.16,
      ttr: 0.05,
      avgSentenceLength: 0.11,
      avgWordLength: 0.07,
      commaDensity: 0.07,
      formalOpening: 0.05,
      bigramRepetitionRate: 0.10,
      functionWordRate: 0.11,
      charEntropy: 0.03,
      punctuationProfile: 0.02,
      paragraphSymmetry: 0.02,
      headingDensity: 0.03,
      questionDensity: 0.02
    };

    let total = 0;
    let weightSum = 0;
    Object.keys(weights).forEach(key => {
      total += (scores[key] || 0) * weights[key];
      weightSum += weights[key];
    });
    const normalized = total / weightSum; // -1..+1
    const baseScore = clamp(Math.round(((normalized + 1) / 2) * 100), 0, 100);

    let finalScore = baseScore;

    const strongAISignals = Object.values(scores).filter(v => v > 0.6).length;
    const mediumAISignals = Object.values(scores).filter(v => v > 0.3).length;

    if (options.mode === "strict") {
      if (strongAISignals >= 3 && finalScore < 75) finalScore = 75;
      if (strongAISignals >= 4 && finalScore < 85) finalScore = 85;
      if (strongAISignals >= 5 && finalScore < 92) finalScore = 92;

      if (strongAISignals >= 2 && mediumAISignals >= 3 && finalScore < 70) {
        finalScore = 70;
      }

      if (
        stats.stdDev <= 3.5 &&
        stats.markerDensity >= 1.5 &&
        stats.functionWordRate <= 0.36 &&
        finalScore < 90
      ) {
        finalScore = 90;
      }

      // Yeni: uzun + başlıklı + soru-cevaplı kurumsal metinlere ekstra itiş
      if (
        stats.totalWords >= 600 &&
        stats.headingDensity >= 0.15 &&
        stats.questionDensity >= 0.05 &&
        normalized > 0 &&
        finalScore < 80
      ) {
        finalScore = 80;
      }

      // Gri bandı AI lehine bük
      if (finalScore >= 45 && finalScore <= 65 && normalized > 0) {
        finalScore = 65;
      }
    }

    let label = "Büyük İhtimalle İnsan Yazımı";
    let color = "#16a34a";

    if (finalScore >= 40 && finalScore < 65) {
      label = "Karışık / İnsan + AI Olabilir";
      color = "#f59e0b";
    }
    if (finalScore >= 65 && finalScore < 80) {
      label = "Büyük İhtimalle Yapay Zekâ";
      color = "#f97316";
    }
    if (finalScore >= 80) {
      label = "Neredeyse Kesin Yapay Zekâ";
      color = "#dc2626";
    }

    const details =
      `AI Skoru (strict v6): ${finalScore} / 100 (ham skor: ${baseScore})\n` +
      `Kelime sayısı: ${stats.totalWords}, Cümle sayısı: ${stats.sentenceCount}\n` +
      `Cümle std sapması: ${stats.stdDev.toFixed(2)} | ` +
      `AI kalıp yoğunluğu: %${stats.markerDensity.toFixed(2)} | ` +
      `TTR: ${stats.ttr.toFixed(3)} | ` +
      `Ø cümle uzunluğu: ${stats.avgSentenceLength.toFixed(1)} kelime | ` +
      `Ø kelime uzunluğu: ${stats.avgWordLength.toFixed(2)} harf\n` +
      `İşlevsel kelime oranı: ${(stats.functionWordRate * 100).toFixed(1)}% | ` +
      `Virgül yoğunluğu: ${stats.commaDensity.toFixed(3)} | ` +
      `Noktalama yoğunluğu: ${stats.punctuationDensity.toFixed(3)} | ` +
      `!/? oranı: ${stats.exqRatio.toFixed(3)}\n` +
      `Bigram tekrar oranı: ${stats.bigramRepetitionRate.toFixed(3)} | ` +
      `Paragraf uzunluğu sapması: ${stats.paragraphStdDev.toFixed(2)} | ` +
      `Karakter entropisi: ${stats.charEntropy.toFixed(3)}\n` +
      `Başlık yoğunluğu: ${(stats.headingDensity * 100).toFixed(1)}% | ` +
      `Soru yoğunluğu: ${(stats.questionDensity * 100).toFixed(1)}%\n\n` +
      `Yorumlar:\n- ` + comments.join("\n- ");

    return {
      score: finalScore,
      rawScore: baseScore,
      label,
      color,
      mode: options.mode || "strict",
      details,
      stats,
      raw: { scores }
    };
  }

  return { detect };
})();
